#pragma rtGlobals=1					// Use modern global access method.
#include <File Name Utilities>	// WaveMetrics procedure - provides RemoveFilePath utility

// Dynamic Clamp routine using ITC18 XOPs						Last edit: 15 Feb 2022
// by John Bekkers

// **********
// PURPOSE
// **********

// This procedure runs a dynamic clamp to inject an EPSG template recorded 'in vivo' while simultaneously
// triggering an extracellular stimulator at times determined by APs recorded from NG and HZ cells 'in vivo'.
// The EPSC and AP data are read from AxoGraph files, assuming column 1 contains the EPSCs/APs and column 2
// contains the respiration trace.  The EPSC and AP episodes are aligned at the first upward transient in the
// respiration trace after odour onset, then the AP latencies in each respiration epoch of the AP episode are
// linearly morphed to match the corresponding respiration epoch of the EPSC episode.  The EPSG and stimulator
// can be turned on or off as required.  Note that the dynamic clamp runs only at 50 kHz, so the timebase of
// the sampled data (typically 20 kHz) has to be stretched to the higher rate.

// **********
// SOFTWARE REQUIREMENTS
// **********

// 1. Igor Pro 6 or higher, available from https://www.wavemetrics.com/
// 2. ITC18 XOPs (Igor External Operations), available from https://www.heka.com/ and installed
//		according to their instructions.

// **********
// HARDWARE REQUIREMENTS
// **********

// 1. Instrutech ITC18 digitizing interface.
//		NOTE: The successor to the ITC18 (HEKA LIH 8+8) has NOT been tested with this code.
// 2. MultiClamp 700A or B, or similar patch clamp amplifier.
// 3. Apple Mac computer with Intel chip or Motorola chip (PowerPC).
//		This procedure has not been tested on a Mac with an M1 chip or on a Windows PC.  Modifications to this code
//		may be required to run on these computers.

// **********
// DYNAMIC CLAMP
// **********

// The dynamic clamp part of this procedure uses the command 'ITC18RunDynamicClamp' to calculate:
//		Io(t) = Ge*excitWave(t)*(E_Erev-Vm(t)) + Gi*inhibWave(t)*(I_Erev-Vm(t)) + Gc*forceWave(t)

// 	where 
//		Io(t) = command current to cell in Amps, from DAC0
//		Vm(t) = membrane potential of cell in Volts, into ADC0
//		excitWave(t) and inhibWave(t) contain the excitatory and inhibitory conductances (in Siemens), respectively
//		forceWave(t) is an additional forcing waveform
//		Gc, Ge and Gi are additional gains, set to 1 here
//		E_Erev and I_Erev are the excitatory and inhibitory reversal potentials (in Volts), respectively

// The following waves are used by the dynamic clamp (where "EpiLen" is the length of the wave containing the data):
//		forceWave		// Double precision, EpiLen length - contains forcing template
//		excitWave		// Double precision, EpiLen length - contains EPSG template
//		inhibWave		// Double precision, EpiLen length - contains IPSG template
//		DAC1Wave			// Single precision, EpiLen length - DAC1 output - NOT USED HERE
//		DAC2Wave			// Single precision, EpiLen length - DAC2 output
//		DIGWave			// Single precision, EpiLen length - DIG0 output
//		sampWave			// Single precision, EpiLen*4 length - entangled ADC0, 1, 2, 3
							// Note that ADC3 and ADC4 are not used here.  They can be used for sampling other signals

// Note that all of the above waves must be explicitly included in the ITC18 XOP command 'ITC18RunDynamicClamp'
// but only the excitatory conductance is applied here.  Hence, the waves 'forceWave' and 'inhibWave' are
// set to zero (but must be defined and included in the argument list).

//	The following waves are used to obtain the result of the dynamic clamp:
//		VmWave			// Single precision, EpiLen length - disentangled ADC0 from sampWave
//		IoWave			// Single precision, EpiLen length - disentangled ADC1 from sampWave

// Physical setup (NB: these channel assignments are hard-wired in the Igor XOP):
//		(i) DAC0 to COMMAND input on MultiClamp
//		(ii) ADC0 from SCALED OUTPUT on MultiClamp
//		(iii) ADC1 from RAW OUTPUT on MultiClamp (or can be used to sample other inputs, if desired)
//		(iv) DIG0 to stimulator

//		NB: Arguments to 'ITC18RunDynamicClamp' can be local variables, except E_ERev and I_ERev, which have to be globals.

// **********
// USAGE
// **********

// 1. Install ITC18 XOPs (but code will automatically run in dummy mode if XOPs or hardware are absent).
// 2. Open this procedure in Igor and Compile.
// 3. Under Igor > Macros select 'Dynamic In Vivo'.
// 4. Use the GUI to control the procedure, as detailed below.

// **********
// GUI
// **********

// ------------------------------------
// "Main" tab:
// ------------------------------------
// Evoke				Runs the dynamic clamp, using the currently loaded excitatory postsynaptic conductance (EPSG) and stimulus
//						waves.  The start of each episode (default duration 10s) is indicated by 1 tone, the end by 2 tones.
//						(In dummy mode the duration is zero).  The dynamic clamp loops 'Num Epis' times.  Press Command-period
//						to stop prematurely.  Note that, in a real experiment, the actions under the 'Setup' tab must be done first.
//						See 'Setup' tab below for more details.
// Episode			Episode counter (read-only).
// RepInt (s)		Start-to-start time between episodes.
// Num Epis			Total number of episodes to acquire.
// Peak EPSG (nS)	Peak of excitatory postsynaptic conductance, calculated from EPSC data file (read-only).
//						Value is set in "g's" tab.
// Display Io		Checkbox to display the current injected by the dynamic clamp.
// Save epis		Checkbox to save the Vm data.  Data are saved (appended) at the end of each episode, so stopping prematurely
//						does not lose any previous episodes.
// # Saved			Number of episodes saved (read-only).
// R					Reset file pointer to zero.  This allows you to overwrite all previous data appended to the current file.
// New File			Runs a dialog to open a new file.
// Incr File		Automatically increments an integer at the end of the base file name.  This assumes base file name has the
//						format "Name.#" where # = an integer (typically start with '0').
// Data File		Name of file in which the Vm data are being saved (read-only).
// Test pulse		Opens a new window in which a test pulse is running.  To exit this window, press 'escape' to stop the test
//						pulse then click 'Exit'.
// VC / IC			Select whether the test pulse is run in voltage clamp or current clamp mode.

// ------------------------------------
// "Gains" tab:
// ------------------------------------
// SampInt (s)			Sample interval for dynamic clamp (DC), loaded EPSC episode, and loaded action potential (AP) episode
//								with odour response of NG or HZ cell.  These values are either obtained from the loaded data (EPSC, AP)
//								or are hard-wired into the dynamic clamp command (DC).
// EpiLen					Episode length (number of points) for DC, EPSC and AP data traces.
// Vm Gain (MultiClamp)	'Scaled Output' gain for Vm set in the MultiClamp Commander.
// ADC Range (ITC-18)	Programmable range of ADCs in ITC-18.  For example, '10' means a range of +/- 10 Volts.
// Stim Scale (nA/V)		'Stimulus Scaling' setting for IC mode in the MultiClamp Commander.  The larger value (2 nA/V) is
//								normally preferred for a dynamic clamp experiment in which large currents need to be injected.

// ------------------------------------
// "g's" tab:
// ------------------------------------
// EPSC Er (mV)			Assumed reversal potential for the loaded EPSC data.  Used to calculate the conductance template (EPSG).
// Conductance (nS)		Peak-scaled conductance of the EPSG template.  This is normally adjusted at the beginning of the
//								experiment to obtain the desired baseline firing response.

// ------------------------------------
// "Setup" tab:
// ------------------------------------
// Read EPSC File			Opens a dialog to read and plot the odour-evoked EPSC episode (and associated respiration episode)
//								that is used to calculate the EPSG applied by the dynamic clamp.
// EPSC resp threshold	Threshold value used by the procedure to find the peaks in the respiration episode that accompanies
//								the EPSC episode.
// Read AP File			Opens a dialog to read and plot the odour-evoked response of a selected NG or HZ cell, together with
//								the corresponding respiration episode.  These data are used for calculating the times at which to
//								trigger the extracellular stimulator. 
// AP resp threshold		Threshold value to find the peaks in the respiration episode that accompanies each interneuron AP episode.
// AP peak threshold		Threshold value (in mV) for detecting APs in the interneuron episode.
// Morph						Runs a function that adjusts the timing of the APs in the interneuron episode in order to match the
//								interneuron's respiration pattern to that of the EPSC episode.  See paper and code for details.
// Odour start (s)		Start time for odour onset in the 'in vivo' experiment.
// Baseline duration (s)	Duration of baseline before odour onset, used to calculate the dynamic clamp conductance template.
// DC episode duration (s)	Duration of entire dynamic clamp conductance template, and hence the duration of the recorded Vm episode.
// EPSG ON					Checkbox to turn on the injected excitatory postsynaptic conductance (EPSG).
// Stimulator ON			Checkbox to turn on the extracellular stimulator, which applies stimuli at times determined by the 
//								currently loaded interneuron data.
// Plot DC Waves ON		Checkbox to display the EPSG and morphed stimulus trigger pattern that are applied.
// Clear All Plots		Clears all plots opened in this tab, leaving just the main Vm plot.


Menu "Macros"
	"Dynamic In Vivo", div()
End

Function div()
	InitGlobals()			// Initialise all global variables.
	InitDCWaves()			// Define waves.
	MakeControlPanels()	// Draw GUI.
End

//_________ Initialise all globals __________

Function InitGlobals()
	
	if( exists("ITC18Reset") == 0 )			// Returns zero if 'ITC18Reset' (command from ITC18 XOP) does not exist.
		Variable/G ITCisConnected = 0		// = 0 if hardware is not connected, = 1 if hardware is connected.
	else
		Variable/G ITCisConnected = 1
	endif

	// Dynamic Clamp globals (except reversal potentials, which are under 'user interface').
	
	// The following interface channel assignments cannot be changed with the current XOP.
		
	Variable/G CommandDAC = 0			// DAC output to COMMAND input on Multiclamp.
	Variable/G ScaledOutputADC = 0		// ADC input from SCALED OUTPUT on Multiclamp.
	Variable/G RawOutputADC = 1			// ADC input from RAW OUTPUT on Multiclamp.
	
	Variable/G SampleClockTicks = 4		// 4 = 50 kHz sampling rate per ADC channel (i.e. 20 s samp int).
												// NOTE: This is the default sample rate for the dynamic clamp function
												// and cannot be changed.
	Variable/G DCSampInt = 20			// Sample interval in s.
	
	Variable/G DCEpiLen = 500000		// Episode length (10 s) - this is changed to length specified in 'Setup' tab.
	Variable/G DCEpiLenSecs				// Dynamic clamp episode length (s).
	DCEpiLenSecs = DCEpiLen*DCSampInt*1e-6
	
	Variable/G AcqFlags = 2				// Same as used in 'ITC18StartAcq' - start acquiring immediately.
	Variable/G ADCRange = 10				// Programmable gain in ITC18 (10 gives 100 mV, 5 gives 200 mV).
		
	Variable/G VmGain = 10				// Gain in MultiClamp for Vm.  Make sure this is set in Multiclamp.
	Variable/G StimScaling = 2000e-12	// In Siemens, equivalent to 2000 pA/V in MultiClamp.
	Variable/G Gc = 1						// Forcing gain.
	Variable/G Ge = 1						// Excitatory gain.
	Variable/G Gi = 1						// Inhibitory gain.
	
	// Globals used by file operations
	
	if(stringmatch(IgorInfo(4), "Intel"))
		Variable/G ChipFlag = 1			// 1 = Intel (Intel chip Macs); required in FBinRead.
	else
		Variable/G ChipFlag = 0			// 0 = PowerPC (Motorola chip).
	endif
	
	// Globals used by "Test Pulse"
	
	Variable/G TestPulseVCorIC = 1		// = 1 if VC mode, = 2 if IC mode.
	Variable/G TestPulseFirstVisit = 1	// Flag to tell test pulse to use default parameters on first startup.
	Variable/G TPSampInt = 10								// Test Pulse sample interval (s) (only used in next line).
	Variable/G	 TPSampleClockTicks = TPSampInt/1.25	//  Test Pulse sampling in clock ticks (for ITC-18).
	
	Variable/G VCHeadStageGain = 0.5	// In V/nA; default is for Rf = 500 MOhm.
	Variable/G VCScaledOutputGain = 2	// VC Scaled Output Gain for Channel 1 on MultiClamp.
	Variable/G VCExtCmdSens = 20		// (mV/V) VC External Command Sensitivity for Channel 1.
	
	Variable/G ICScaledOutputGain = 10	// IC Scaled Output Gain for Channel 1 on MultiClamp.
	Variable/G ICExtCmdSens = 400		// (pA/V) IC External Command Sensitivity for Channel 1.
	
	// Global variables and flags used by the user interface.
	
			// Main Tab

	Variable/G	RepInt = 20				// Interval between successive stimuli, in sec.
	Variable/G	LastTime					// Remember time of last evoke, for calculating waiting time.
	Variable/G	NumTrials = 10			// Number of trials.
	Variable/G	EpisodeCounter = 0		// Counter for total number of episodes.
	Variable/G	DisplayIoIsOn = 0		// 'Display Io' checkbox.
	Variable/G	SaveEpisIsOn = 0		// 'SaveEpis' checkbox.
	Variable/G	NumberSaved = 0			// Number of episodes saved.
	String/G	FileName = ""				// Data file name with entire path.
	String/G	ShortFileName = ""			// Short file name for display.  NB: Also displays EPSC and AP file names.
	
			// g's Tab
	
	Variable/G E_Erev = 10				// EPSC reversal potential, in mV.
	Variable/G E_MeanAmp = 1				// Mean excitatory conductance, in nS.
	Variable/G I_Erev = 0					// This is not used, but it needs to go into argument list for dynamic clamp.
	Variable/G DisplayGsIsOn = 0		// 'Display Io' checkbox.
	
			// Setup Tab
	
	Variable/G EPSCRespThresh = 0.6		// Threshold for detecting respiration peaks for EPSCs.
	Variable/G EPSCbaselinems1 = 2500	// Baseline EPSC trace over range EPSCbaselinems1 to EPSCbaselinems2.
	Variable/G EPSCbaselinems2 = 3000
	Variable/G APRespThresh = 0.6		// Threshold for detecting respiration peaks for APs.
	Variable/G APPeakThresh = -20		// Threshold for detecting AP peaks.
	Variable/G RespMinWidthX = 200		// Skip this time (ms) before searching for next resp peak.
	Variable/G APMinWidthX	= 3			// Skip this time (ms) before searching for next AP peak.
	Variable/G MaxPeaks = 200			// Maximum number of respiration & AP peaks that can be found.
	
	Variable/G OdourStartTime = 8		// Odour start time (s).
	Variable/G BaselineDur = 2			// Baseline before odour start (s).
	Variable/G EPSCwaveFirstOdourResp_index = 0		// Origin of EPSCwave (at first respiration peak after odour start).
	
	Variable/G EPSGIsOn = 1				// Checkbox for switching on excitatory conductance.
	Variable/G StimIsOn = 1				// Checkbox for switching on stimulator.
	Variable/G DCWavesPlotIsOn = 0		// Checkbox for plotting stimulus waves passed to dynamic clamp.
	
	Variable/G SampInt = 0				// SampInt used by ReadAGXFile.
	Variable/G EpiLen = 0					// EpiLen used by ReadAGXFile.
	Variable/G EPSCSampInt = 0			// Ditto for each type of episode (EPSC or AP).
	Variable/G EPSCEpiLen = 0
	Variable/G APSampInt = 0
	Variable/G APEpiLen = 0
	
	// Display the main plot window
	
	DoWindow/K Current					// Close old plot windows.
	DoWindow/K EPSCPlot
	DoWindow/K APPlot
	DoWindow/K MorphedAPPlot
	DoWindow/K DCWavePlot
	DoWindow/F DynamicClamp			// Bring main panel to front.

End

//_________ Initialise waves used by dynamic clamp __________

Function InitDCWaves()
	
	NVAR DCEpiLen, DCSampInt
	
	Make/O/D/N=(DCEpiLen) forceWave, excitWave, inhibWave	// Double Precision waves.
	Make/O/N=(DCEpiLen) DAC2Wave, DIGWave							// Single Precision waves.
	Make/O/N=(DCEpiLen) VmWave, IoWave, ADC2Wave, ADC3Wave
	Make/O/N=(DCEpiLen*4) sampWave
//	Make/O/N=(DCEpiLen) DAC1Wave			// (Not using DAC1 here).
	SetScale/P x 0, DCSampInt/1000,"ms", VmWave, IoWave
	SetScale/P x 0, DCSampInt/1000,"ms", forceWave, excitWave, inhibWave
	SetScale/P x 0, DCSampInt/1000,"ms",  DIGWave

	forceWave = 0
	excitWave = 0
	inhibWave = 0
//	DAC1Wave = 0
	DAC2Wave = 0
	DIGWave = 0
	VmWave = 0
	IoWave = 0
	sampWave = 0
	
	DoWindow/K Voltage							// Close old Vm window.
	Display/M/W=(9.5,1.5,30.5,15.5) VmWave
	DoWindow/C Voltage							// Set name (used for closing window).
	SetAxis left -100, 60						// Change axis to -100 mV to +60 mV.
	SetScale y 0, 0, "mV", VmWave

	DoWindow/F DynamicClamp

End

//_________ Routine for drawing the control panels __________

Function MakeControlPanels()
	String	ctrlName = "", popStr=""
	
	DoWindow/K DynamicClamp
	NewPanel /W=(20,53,250,526) as "DynamicClamp"
	DoWindow/C DynamicClamp
	MakeMainTab()								// Define controls in each of the 4 Tabs.
	MakeGainsTab()
	MakegTab()
	MakeSetupTab()
	SelectTabs(ctrlName, 0)					// Select the 'Main' Tab.
End

//_________ Draw "Main" tab __________

Function MakeMainTab()	
	NVAR	RepInt
	NVAR	NumTrials
	NVAR	EpisodeCounter
	NVAR	E_MeanAmp
	NVAR	DisplayIoIsOn
	NVAR	SaveEpisIsOn
	NVAR	NumberSaved
	SVAR	ShortFileName
	
	TabControl tabcon0_0,pos={5,9},size={220,454},tabLabel(0)="Main",value= 0
	TabControl tabcon0_0,tabLabel(1)="Gains",tabLabel(2)="g's",tabLabel(3)="Setup"
	
	Button button0_0,pos={76,48},size={68,24},proc=EvokeButton,title="Evoke"
	ValDisplay valdisp0_1,pos={48,86},size={110,14},title="Episode"
	ValDisplay valdisp0_1,limits={0,0,0},barmisc={0,1000},value= #"EpisodeCounter"
	SetVariable setvar0_2,pos={60,114},size={116,15},title="RepInt (s)"
	SetVariable setvar0_2,limits={0,inf,1},value= RepInt
	SetVariable setvar0_3,pos={59,145},size={120,15},title="Num Epis"
	SetVariable setvar0_3,value= NumTrials
	CheckBox check0_4,pos={53,205},size={59,15},proc=DisplayIoCheck,title="Display Io"
	CheckBox check0_4,value= 0
	CheckBox check0_5,pos={53,236},size={59,15},proc=SaveEpisCheck,title="Save Epis"
	CheckBox check0_5,value= 0
	ValDisplay valdisp0_6,pos={36,266},size={107,14},title="# Saved"
	ValDisplay valdisp0_6,limits={0,0,0},barmisc={0,1000},value= #"NumberSaved"
	Button button0_7,pos={75,294},size={78,24},proc=NewFileButton,title="New File"
	SetVariable setvar0_8,pos={26,366},size={175,15},title="Data File"
	SetVariable setvar0_8,value= ShortFileName
	Button button0_9,pos={157,264},size={22,20},proc=ResetFileButton,title="R"
	ValDisplay valdisp0_10,pos={60,177},size={100,14},title="Peak EPSG (nS)"
	ValDisplay valdisp0_10,limits={0,0,0},barmisc={0,1000},value= #"E_MeanAmp"
	Button button0_11,pos={83,329},size={63,25},proc=IncrFileButton,title="Incr File"
	GroupBox group0_12,pos={24,400},size={180,50}
	PopupMenu popup0_13,pos={33,415},size={49,20}
	PopupMenu popup0_13,mode=1,popvalue="VC",value= #"\"VC;IC\"",proc=VC_ICPopup
	Button button0_14,pos={90,412},size={100,25},proc=TestPulseButton,title="Test pulse"

End

//_________ Draw "Gains" tab __________

Function MakeGainsTab()
	NVAR	DCEpiLen, EPSCEpiLen, APEpiLen
	NVAR	DCSampInt, EPSCSampInt, APSampInt
	
	TabControl tabcon0_0,proc=SelectTabs,tabLabel(1)="Gains"
	
	ValDisplay valdisp1_0,pos={88,90},size={35,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_0,value= #"DCSampInt"
	ValDisplay valdisp1_1,pos={148,90},size={70,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_1,value= #"DCEpiLen"
	ValDisplay valdisp1_2,pos={88,120},size={35,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_2,value= #"EPSCSampInt"
	ValDisplay valdisp1_3,pos={148,120},size={70,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_3,value= #"EPSCEpiLen"
	ValDisplay valdisp1_4,pos={88,150},size={35,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_4,value= #"APSampInt"
	ValDisplay valdisp1_5,pos={148,150},size={70,14},limits={0,0,0},barmisc={0,1000}
	ValDisplay valdisp1_5,value= #"APEpiLen"
	PopupMenu popup1_6,pos={31,200},size={79,20},title="Vm Gain (MultiClamp)"
	PopupMenu popup1_6,mode=4,value= #"\"1;2;5;10;20;50;100;200;500;1000 \"",proc=SetVmGainPopup
	PopupMenu popup1_7,pos={31,240},size={77,20},title="ADC Range (ITC-18)"
	PopupMenu popup1_7,mode=4,value= #"\"1;2;5;10\"",proc=SetADCRangePopup
	PopupMenu popup1_8,pos={31,280},size={133,20},title="Stim Scale (nA/V)"
	PopupMenu popup1_8,mode=2,value= #"\".4;2\"",proc=SetStimScalingPopup
End

//_________ Draw "g's" tab __________

Function MakegTab()
	NVAR	E_Erev, E_MeanAmp

	TabControl tabcon0_0,proc=SelectTabs,tabLabel(2)="g's"
	
	SetVariable setvar2_0,pos={58,76},size={122,15},title="EPSC Er (mV)"
	SetVariable setvar2_0,limits={-100,100,1},value= E_Erev
	SetVariable setvar2_1,pos={40,103},size={140,15},title="Conductance (nS)"
	SetVariable setvar2_1,limits={0,200,0.1},value= E_MeanAmp, proc=ChangePeakConductance

End

//_________ Draw "Setup" tab __________

Function MakeSetupTab()
	
	NVAR	EPSCRespThresh, APRespThresh, APPeakThresh
	NVAR	OdourStartTime, BaselineDur, DCEpiLenSecs
	NVAR 	EPSGIsOn, StimIsOn, DCWavesPlotIsOn

	TabControl tabcon0_0,proc=SelectTabs,tabLabel(3)="Setup"
		
	Button button3_0,pos={54,49},size={120,24},proc=ReadEPSCFileButton,title="Read EPSC File"
	SetVariable setvar3_1,pos={44,81},size={140,15},title="EPSC resp threshold"
	SetVariable setvar3_1,limits={0,1,0.1},value= EPSCRespThresh
	Button button3_2,pos={54,121},size={120,24},proc=ReadAPFileButton,title="Read AP File"
	SetVariable setvar3_3,pos={49,156},size={130,15},title="AP resp threshold"
	SetVariable setvar3_3,limits={0,1,0.1},value= APRespThresh
	SetVariable setvar3_4,pos={34,180},size={160,15},title="AP peak threshold (mV)"
	SetVariable setvar3_4,limits={-40,50,10},value= APPeakThresh
	Button button3_5,pos={54,218},size={120,24},proc=MorphButton,title="Morph"
	SetVariable setvar3_6,pos={72,254},size={120,15},title="Odour start (s)"
	SetVariable setvar3_6,limits={0,30,1},value= OdourStartTime
	SetVariable setvar3_7,pos={52,280},size={140,15},title="Baseline duration (s)"
	SetVariable setvar3_7,limits={0,30,0.5},value= BaselineDur,proc=SetBaselineDuration
	SetVariable setvar3_8,pos={35,306},size={157,15},title="DC episode duration (s)"
	SetVariable setvar3_8,limits={0,30,1},value= DCEpiLenSecs,proc=SetDCEpisodeDuration
	CheckBox check3_9,pos={73,335},size={56,15},proc=EPSGOnCheck,value=EPSGisOn,title="EPSG ON"
	CheckBox check3_10,pos={73,360},size={79,15},proc=StimOnCheck,value=StimIsOn,title="Stimulator ON"
	CheckBox check3_11,pos={73,385},size={85,15},proc=PlotDCWavesOnCheck,value=DCWavesPlotIsOn,title="Plot DC Waves ON"
	Button button3_12,pos={54,417},size={120,24},proc=ClearAllPlotsButton,title="Clear All Plots"
	
End

//_________ Routine that erases and redraws each tab when selected __________

Function SelectTabs(name,tab)
	String name
	Variable tab
	
	SetDrawLayer/K UserBack
	
	Button button0_0,disable= (tab!=0)				// "Main" tab.
	ValDisplay valdisp0_1,disable= (tab!=0)
	SetVariable setvar0_2,disable= (tab!=0)
	SetVariable setvar0_3,disable= (tab!=0)
	CheckBox check0_4,disable= (tab!=0)
	CheckBox check0_5,disable= (tab!=0)
	ValDisplay valdisp0_6,disable= (tab!=0)
	Button button0_7,disable= (tab!=0)
	SetVariable setvar0_8,disable= (tab!=0)
	Button button0_9,disable= (tab!=0)
	ValDisplay valdisp0_10,disable= (tab!=0)
	Button button0_11,disable= (tab!=0)
	GroupBox group0_12,disable= (tab!=0)
	PopupMenu popup0_13,disable= (tab!=0)
	Button button0_14,disable= (tab!=0)

	ValDisplay valdisp1_0,disable=(tab!=1)		// "Gains" tab.
	ValDisplay valdisp1_1,disable=(tab!=1)
	ValDisplay valdisp1_2,disable=(tab!=1)
	ValDisplay valdisp1_3,disable=(tab!=1)
	ValDisplay valdisp1_4,disable=(tab!=1)
	ValDisplay valdisp1_5,disable=(tab!=1)
	PopupMenu popup1_6,disable=(tab!=1)
	PopupMenu popup1_7,disable=(tab!=1)
	PopupMenu popup1_8,disable=(tab!=1)

	SetVariable setvar2_0,disable=(tab!=2)		// "g's" tab.
	SetVariable setvar2_1,disable=(tab!=2)

	Button button3_0,disable=(tab!=3)				// "Setup" tab.
	SetVariable setvar3_1,disable=(tab!=3)
	Button button3_2,disable=(tab!=3)
	SetVariable setvar3_3,disable=(tab!=3)
	SetVariable setvar3_4,disable=(tab!=3)
	Button button3_5,disable=(tab!=3)
	SetVariable setvar3_6,disable=(tab!=3)
	SetVariable setvar3_7,disable=(tab!=3)
	SetVariable setvar3_8,disable=(tab!=3)
	CheckBox check3_9,disable=(tab!=3)
	CheckBox check3_10,disable=(tab!=3)
	CheckBox check3_11,disable=(tab!=3)
	Button button3_12,disable=(tab!=3)	
	
	if(tab==1)											// Draw in text, lines, etc.
		SetDrawLayer UserBack
		SetDrawEnv fsize= 10
		DrawText 73,71,"SampInt (s)"
		SetDrawEnv fsize= 10
		DrawText 166,71,"EpiLen"
		SetDrawEnv fstyle= 1
		DrawText 31,103,"DC:"
		SetDrawEnv fstyle= 1
		DrawText 30,132,"EPSC:"
		SetDrawEnv fstyle= 1
		DrawText 31,162,"AP:"
	endif
	
	if(tab==2)
		SetDrawLayer UserBack
		SetDrawEnv fstyle= 2
		DrawText 30,62,"EPSCs"
	endif

End

//_________ FUNCTIONS CALLED BY "Main" TAB ___________

//_________ Hit "Evoke" Button ______________________

Function EvokeButton(ctrlName) : ButtonControl
	String ctrlName
	
	Variable trialCount, templateCount, fileHandle
	String tmpStr, incrFileName, mouseButton
	
	NVAR	ScaledOutputADC, ADCRange, DCEpiLen, SampleClockTicks, AcqFlags
	NVAR	E_Erev, I_Erev, NumTrials
	NVAR	VmGain, StimScaling, Gc, Ge, Gi, EpisodeCounter, SaveEpisIsOn, NumberSaved, LastTime, RepInt
	NVAR	ITCisConnected
	SVAR	FileName, ShortFileName
	WAVE	forceWave, excitWave, inhibWave, VmWave, IoWave, SampWave
	
	DoWindow/F Voltage
	DoWindow/F DynamicClamp
	
	if( ITCisConnected == 1 )
		Execute "ITC18SetADCRange ScaledOutputADC, ADCRange"		// Commands from ITC18 XOP.
		Execute "ITC18LoadDynamicClampMode"
	endif
	
	EpisodeCounter = 0
	for( trialCount=1; trialCount<=NumTrials; trialCount +=1 )
	
		if( EpisodeCounter != 0 )
			do												// If not first episode, wait for 'RepInt' sec.
			while( (ticks-LastTime)/60 < RepInt )
		endif
		LastTime = ticks
		Beep												// Let the user know that the episode is about to start.
		if( ITCisConnected == 1 )					// NIL argument because are not using DAC1Wave.
			Execute "ITC18RunDynamicClamp NIL, DAC2Wave, DIGWave, forceWave, excitWave, inhibWave, sampWave, SampleClockTicks, AcqFlags, 0, ADCRange, VmGain, StimScaling, E_Erev/1000, I_Erev/1000, Gc, Ge, Gi"
		endif	
															// The following lines disentangle the data from the sampled wave
															// generated by 'ITC18RunDynamicClamp'.
		VmWave[0, DCEpiLen-1] = sampWave[4*p+2]			// ADC0 = Vm from clamp.
		IoWave[0, DCEpiLen-1] = sampWave[4*p+3]			// ADC1 = Io to clamp.
//			ADC2Wave[0, DCEpiLen-1] = sampWave[4*p]		// How to get access to other ADC inputs, if required.
//			ADC3Wave[0, DCEpiLen-1] = sampWave[4*p+1]

		VmWave = VmWave*(10240/32768/VmGain/ADCRange)			// Converts to mV.
		IoWave = -IoWave*(10.240/32768)*StimScaling*1e12		// Converts to pA.
		
		if( SaveEpisIsOn )								// If want to save data ...
																	// Write as 32 bit IEEE FP format into 1 file.
			Open/A fileHandle as FileName					// NB: Data is always APPENDED to existing file contents.
			FBinWrite/F=4 fileHandle, VmWave				// Only save Vm.
			Close fileHandle
																	// Alternatively, write in Igor wave format into many files.
//			incrFileName = FileName + num2istr(NumberSaved+1) + ".bwav"
//			Save/O VmWave as incrFileName
//			ShortFileName = RemoveFilePath(incrFileName)
				
			NumberSaved += 1
		endif
		EpisodeCounter += 1
		DoUpdate
		Beep				// Let the user know the episode has ended by beeping twice.
		Sleep/S 0.3
		Beep
	endfor
	
End

//_________ Select "Display Io" Checkbox _______________

Function DisplayIoCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	
	NVAR	DisplayIoIsOn

	if (checked)
		DisplayIoIsOn = 1
		Display/M/W=(9.5,17,30.5,28) IoWave	// Make new graph to display 'Io'.
		DoWindow/C Current							// Set name (used for closing window).
		SetScale y 0, 0, "pA", IoWave
		DoWindow/F DynamicClamp
	else
		DoWindow/K Current							// Kill 'Io' graph window.
		DisplayIoIsOn = 0
	endif

End

//_________ Select "Save Epis" Checkbox _______________

Function SaveEpisCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	
	NVAR	SaveEpisIsOn
	NVAR	EpisodeCounter

	if (checked)
		SaveEpisIsOn = 1
		EpisodeCounter = 0
	else
		SaveEpisIsOn = 0
	endif

End

//_________ Hit "New File" Button ______________________

Function NewFileButton(ctrlName) : ButtonControl
	String ctrlName

	Variable fileHandle
	
	NVAR	EpisodeCounter
	NVAR	NumberSaved
	SVAR	FileName
	SVAR	ShortFileName
	
	NumberSaved = 0						// File Store counter initialised when new file.
	EpisodeCounter = 0					// Zero episode counter.
	Open/D fileHandle					// Dialog only; file not opened.
	if( strlen(S_fileName) != 0 )	// Check if user cancelled.
		FileName = S_fileName
		ShortFileName = RemoveFilePath( FileName )
											// Following line used if are saving in Igor format.
//		ShortFileName = RemoveFilePath(FileName) + num2istr(NumberSaved+1) + ".bwav"
	endif
End

//_________ Hit "Incr File" Button ______________________

// This assumes file name has the format "Name.#" where # = an integer (typically start with '0').

Function IncrFileButton(ctrlName) : ButtonControl
	String ctrlName

	Variable fileHandle
	Variable t1, t2, tempdigit
	
	NVAR	SaveEpisIsOn
	NVAR	EpisodeCounter
	NVAR	NumberSaved
	SVAR	FileName
	SVAR	ShortFileName
	
	if( SaveEpisIsOn )
		t1 = strsearch(FileName, ".", 0)				// Returns location of "." (starting counting at 0).
		t2 = strlen(FileName)								// Returns length of filename (starting counting at 1).
	
		if( (t2-t1) == 2 )									// Single digit after "." (0-9).
			tempdigit = str2num( FileName[t1+1] )		// Strip off last digit and convert to num.
			tempdigit += 1									// Increment number.
		elseif( ( t2-t1) == 3 )							// Double digit after "." (>9).
			tempdigit = str2num( FileName[t1+1, t1+2] )
			tempdigit += 1									// Increment number.
		endif

		FileName = FileName[0, t1]						// Strip digits after ".".
		FileName[10000] = num2str(tempdigit)			// Append new incremented digit.
	
		ShortFileName = RemoveFilePath( FileName )
		NumberSaved = 0										// Zero save number counter.
		EpisodeCounter = 0									// Zero episode counter.
	else
		Beep		// Error: Can't increment file name if not saving data.
	endif
	
End

//_________ Hit "R" Button to reset file _____________

// This resets the file pointer of the current file to zero.
// (Useful because data are always APPENDED to data file).

Function ResetFileButton(ctrlName) : ButtonControl
	String ctrlName

	Variable fileHandle

	NVAR	EpisodeCounter
	NVAR	NumberSaved
	SVAR	FileName, ShortFileName
	
	Open/A fileHandle as FileName	// Get file handle by opening file.
	FSetPos fileHandle, 0				// Set file pointer to start of file.
	Close fileHandle					// Close file.
	
											// Following line used if are saving in Igor format.
//	ShortFileName = RemoveFilePath(FileName) + num2istr(NumberSaved+1) + ".bwav"
			
	NumberSaved = 0						// Zero save number counter.
	EpisodeCounter = 0					// Zero episode counter.
End

//_________ Test pulse recording mode popup _____________

Function VC_ICPopup (ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName
	Variable popNum
	String popStr
	
	NVAR TestPulseVCorIC
	
	TestPulseVCorIC = popNum
	
End

//_________ FUNCTIONS CALLED BY "Gains" TAB ___________

//_________ Set Current Clamp 'VmGain' on MultiClamp ________

Function SetVmGainPopup(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName
	Variable popNum
	String popStr
			// List of allowable gains.
	Make/O/N=11 VmGainArray={ 0,1,2,5,10,20,50,100,200,500,1000 }
	
	NVAR	VmGain
	
	VmGain = VmGainArray[popNum]
End

//_________ Set ADC Range on ITC18 ___________________________

Function SetADCRangePopup(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName
	Variable popNum
	String popStr
			// List of allowable ADC ranges (in Volts).
	Make/O/N=4 ADCRangeArray={ 0,1,2,5,10 }
	
	NVAR	ScaledOutputADC, ADCRange
	NVAR	ITCisConnected

	ADCRange = ADCRangeArray[popNum]
	if( ITCisConnected == 1 )
		Execute "ITC18SetADCRange ScaledOutputADC, ADCRange"
	endif
End

//_________ Set Current Clamp 'StimScaling' on MultiClamp _________

Function SetStimScalingPopup(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName
	Variable popNum
	String popStr
			// List of allowable 'StimScaling' settings on MultiClamp (in A/V).
	Make/O/N=2 StimScalingArray={ 0,400e-12, 2000e-12 }
	
	NVAR	StimScaling
	
	StimScaling = StimScalingArray[popNum]
End

//_________ FUNCTIONS CALLED BY "g's" TAB ___________

//_________ Set peak EPSG conductance Variable Control ___________

Function ChangePeakConductance(ctrlName, varNum, varStr, varName) : SetVariableControl
	String ctrlName
	Variable varNum				// This is the set value.
	String varStr
	String varName
	
	NVAR	E_MeanAmp
	
	E_MeanAmp = varNum

	CalculateDCwaves()			// Calculate wave arguments for dynamic clamp.  Function defined below.
	
End

//_________ FUNCTIONS CALLED BY "Setup" TAB ___________

//_________ Press "Read EPSC File" button  ____________

Function ReadEPSCFileButton(ctrlName) : ButtonControl
	String ctrlName
	
	Variable index, ePSCEpiDur, fractionalPeakLocation
	
	NVAR SampInt, EpiLen, EPSCSampInt, EPSCEpiLen, EPSCRespThresh, RespMinWidthX
	NVAR EPSCbaselinems1, EPSCbaselinems2
	
	ReadAxoGraph()									// Function defined below.  Reads 2 columns into temp waves and variables.
	WAVE/Z dataWave									// WAVE declaration must come after 'dataWave' is created.
	
	EPSCSampInt = SampInt							// Assign to values returned by ReadAxoGraph().
	EPSCEpiLen = EpiLen
	ePSCEpiDur = EPSCSampInt*EPSCEpiLen/1000	// Duration of EPSC episode (ms), used for plotting.
		
	Make/O/N=(EPSCEpiLen) EPSCwave
	EPSCwave = dataWave[p][1]					// Assume EPSC is in column 1 returned from 'ReadAxoGraph'.
	SetScale/P x 0, EPSCSampInt/1000,"ms", EPSCwave	// Set x interval for data in EPSCwave.
	Duplicate/O EPSCwave, tmpEPSCwave
	Smooth/B=200/F 2000, tmpEPSCwave			// Smooth heavily to get low freq component.
	WaveStats/Q/R=(EPSCbaselinems1, EPSCbaselinems2) tmpEPSCwave
	EPSCwave = EPSCwave - V_avg					// Subtract mean over first 5 sec to baseline trace.
	
	DoWindow/K EPSCPlot							// Close old EPSC plot.
	Display/M/W=(9,1.5,40,9.5)					// Make new window for EPSC plot.
	DoWindow/C EPSCPlot							// Set name (used for closing window).
	Display/W=(0,0,1,0.6)/HOST=# EPSCwave		// Create sub-window under the current host (hence: #).
	SetScale y 0, 0, "A", EPSCwave
	
	Make/O/N=(EPSCEpiLen) EPSCrespwave
	EPSCrespwave = dataWave[p][2]				// Assume EPSC respiration is in column 2 returned from 'ReadAxoGraph'.
	SetScale/P x 0, EPSCSampInt/1000,"ms", EPSCrespwave
	
	FilterAndNormaliseResp(EPSCrespwave)
	FindEpiPeaks(EPSCrespwave, EPSCSampInt, EPSCRespThresh, RespMinWidthX)		// Function defined below.
	WAVE lat											// 'lat' is a wave containing peak latencies found by 'FindEpiPeaks'.
	Duplicate/O lat, EPSCrespLatencyWave

	Display/W=(0,0.6,1,1)/HOST=## EPSCrespwave	// Create sub-window in host of current sub-window (hence: ##).
	SetScale y 0, 0, "norm", EPSCrespwave
	
	for( index=0; index<numpnts(EPSCrespLatencyWave); index=index+1 )		// Draw lines to mark peaks found.
		fractionalPeakLocation = EPSCrespLatencyWave[index]/ePSCEpiDur
		SetDrawEnv linefgc=(0,65535,0)
			// In the next line 'EPSCPlot#G1' means graph 1 in this plot window; G0 is first sub-window, G1 is second.
		DrawLine/W=EPSCPlot#G1 fractionalPeakLocation, 0, fractionalPeakLocation, 0.1
	endfor
	
	KillWaves/Z dataWave, tmpEPSCwave, lat		// Clean up.
	DoWindow/F DynamicClamp

End

//_________ Press "Read AP File" button  ____________

Function ReadAPFileButton(ctrlName) : ButtonControl
	String ctrlName
	
	Variable index, aPEpiDur, fractionalPeakLocation
	
	NVAR SampInt, EpiLen, APSampInt, APEpiLen, APRespThresh, RespMinWidthX
	
	ReadAxoGraph()								// Function defined below. Reads 2 columns into temp waves and variables.
	WAVE/Z dataWave								// WAVE declaration must come after 'dataWave' is created.
	
	APSampInt = SampInt						// Assign to values returned by ReadAxoGraph().
	APEpiLen = EpiLen
	aPEpiDur = APSampInt*APEpiLen/1000		// Duration of AP episode (ms), used for plotting.
		
	Make/O/N=(APEpiLen) APwave
	APwave = dataWave[p][1]					// Assume AP is in column 1 returned from 'ReadAxoGraph'.
	SetScale/P x 0, APSampInt/1000,"ms", APwave
	DoWindow/K APPlot							// Close old AP plot.
	Display/M/W=(9,10.5,40,18.5)			// Make new AP plot.
	DoWindow/C APPlot							// Set name (used for closing window).
	Display/W=(0,0,1,0.6)/HOST=# APwave
	SetScale y 0, 0, "mV", APwave
	
	Make/O/N=(APEpiLen) APrespwave
	APrespwave = dataWave[p][2]				// Assume AP respiration is in column 2 returned from 'ReadAxoGraph'.
	SetScale/P x 0, APSampInt/1000,"ms", APrespwave
	
	FilterAndNormaliseResp(APrespwave)
	FindEpiPeaks( APrespwave, APSampInt, APRespThresh, RespMinWidthX )	// Function defined below.
	WAVE lat
	Duplicate/O lat, APrespLatencyWave
	
	Display/W=(0,0.6,1,1)/HOST=## APrespwave
	SetScale y 0, 0, "norm", APrespwave
	
	for( index=0; index<numpnts(APrespLatencyWave); index=index+1 )		// Draw lines to mark peaks found.
		fractionalPeakLocation = APrespLatencyWave[index]/aPEpiDur
		SetDrawEnv linefgc=(0,65535,0)
		DrawLine/W=APPlot#G1 fractionalPeakLocation, 0, fractionalPeakLocation, 0.1
	endfor
	
	KillWaves/Z dataWave, lat				// Clean up.
	DoWindow/F DynamicClamp

End

//_________ Press "Morph" button  ____________

Function MorphButton(ctrlName) : ButtonControl
	String ctrlName
	
	Variable index, fractionalPeakLocation, aPEpiDur
	Variable counterEPSCresp, counterAPresp, counterAP
	Variable EPSCwaveFirstPostOdourResp_ms
	Variable currentEPSCRespCycleStart_pnts, currentEPSCRespCycleEnd_pnts
	Variable currentAPRespCycleStart_pnts, currentAPRespCycleEnd_pnts
	Variable durCurrentEPSCCycle_pnts, durCurrentAPCycle_pnts
	
	NVAR APSampInt, APEpiLen, EPSCSampInt, EPSCEpiLen, APPeakThresh, APMinWidthX
	NVAR OdourStartTime, EPSCDur, EPSCwaveFirstOdourResp_index
	WAVE EPSCrespLatencyWave, APrespLatencyWave, EPSCwave, APwave

	FindEpiPeaks(APwave, APSampInt, APPeakThresh, APMinWidthX)		// Function defined below.
	WAVE lat										// 'lat' is a wave containing peak latencies found by 'FindEpiPeaks'.
	Duplicate/O lat, APLatencyWave
	
	aPEpiDur = APSampInt*APEpiLen/1000		// Duration of AP episode (ms), used for plotting.
	
	for( index=0; index<numpnts(APLatencyWave); index=index+1 )		// Draw lines to mark AP peaks.
		fractionalPeakLocation = APLatencyWave[index]/aPEpiDur
		DrawLine/W=APPlot#G0 fractionalPeakLocation, 0.9, fractionalPeakLocation, 1
	endfor
	DoUpdate
	
	index = 0						// Find first EPSC respiration peak after OdourStartTime (and < number of such peaks).
	do
		index = index+1
	while( (EPSCrespLatencyWave[index] < OdourStartTime*1000) && (index<numpnts(EPSCrespLatencyWave)) )
	counterEPSCresp = index	// Save index of this first respiration peak in 'EPSCrespLatencyWave'.
	
									// Save time in 'EPSCwave' at which this first respiration transient occurs.
	EPSCwaveFirstPostOdourResp_ms = EPSCrespLatencyWave[index]
									// Also save as index number - used as the time origin in 'CalculateDCwaves'.
	EPSCwaveFirstOdourResp_index = EPSCwaveFirstPostOdourResp_ms/(EPSCSampInt*1e-3)
	
	index = 0						// Find first AP respiration peak after OdourStartTime (and < number of such peaks).
	do
		index = index+1
	while( (APrespLatencyWave[index] < OdourStartTime*1000) && (index<numpnts(APrespLatencyWave)) )
	counterAPresp = index		// Save index of this first respiration peak in 'APrespLatencyWave'.

	index = 0						// Find first AP peak after first AP respiration peak (and < number of such APs).
	do
		index = index+1
	while( (APLatencyWave[index] < APrespLatencyWave[counterAPresp]) && (index<numpnts(APLatencyWave)) )
	counterAP = index				// Save index of this first AP peak in 'APLatencyWave'.
	
	KillWaves/Z morphedAPWave	// Zero wave into which will morph 'APwave'.
	
	do			// Main loop cycles through counters for AP & EPSC respiration peak times, checking that both of them are within range.
	
									// First, find start, end & duration of current respiration cycle for EPSCs and APs...
		currentEPSCRespCycleStart_pnts = round(EPSCrespLatencyWave[counterEPSCresp]/(EPSCsampInt*1e-3))
		currentEPSCRespCycleEnd_pnts = round(EPSCrespLatencyWave[counterEPSCresp+1]/(EPSCsampInt*1e-3))
		durCurrentEPSCCycle_pnts = currentEPSCRespCycleEnd_pnts - currentEPSCRespCycleStart_pnts
		
		currentAPRespCycleStart_pnts = round(APrespLatencyWave[counterAPresp]/(APsampInt*1e-3))
		currentAPRespCycleEnd_pnts = round(APrespLatencyWave[counterAPresp+1]/(APsampInt*1e-3))
		durCurrentAPCycle_pnts = currentAPRespCycleEnd_pnts - currentAPRespCycleStart_pnts
									// Next, make temp waves the same length as current resp cycle, for both EPSCs and APs...
		Make/O/N=(durCurrentEPSCCycle_pnts) tmpEPSCcycleWave
		Make/O/N=(durCurrentAPCycle_pnts) tmpAPcycleWave
		tmpAPcycleWave = APwave[currentAPRespCycleStart_pnts+p]		// Clip out the AP segment for the current resp cycle...
									// Now interpolate the AP clip into the wave that's the same length as the corresponding EPSC resp cycle...
		tmpEPSCcycleWave = tmpAPcycleWave[p*(numpnts(tmpAPcycleWave)-1)/(numpnts(tmpEPSCcycleWave)-1)]
									// Finally, concatenate the interpolated AP data onto the end of 'morphedAPWave'.
		Concatenate/NP {tmpEPSCcycleWave},  morphedAPWave	// NB: 'Concatenate' creates target wave if it doesn't already exist.
									// Increment the respiration cycle counters and do it all again...
		counterEPSCresp = counterEPSCresp + 1
		counterAPresp = counterAPresp + 1
									// ... until we run out of either EPSC or AP respiration cycles.
	while( (counterEPSCresp<numpnts(EPSCrespLatencyWave)-1) && (counterAPresp<numpnts(APrespLatencyWave)-1) )
	
	SetScale/P x 0, EPSCSampInt/1000,"ms", morphedAPWave
	
	FindEpiPeaks(morphedAPwave, APSampInt, APPeakThresh, APMinWidthX)		// Function defined below.
	WAVE lat										// 'lat' is a wave containing peak latencies found by 'FindEpiPeaks'.
	Duplicate/O lat, morphedAPLatencyWave
													// Add offset to first post-odour EPSC respiration transient (our origin).
	morphedAPLatencyWave = morphedAPLatencyWave + EPSCwaveFirstPostOdourResp_ms

	Make/O/N=(EPSCEpiLen) stimWave
	stimWave = 0
	SetScale/P x 0, EPSCSampInt/1000,"ms", stimWave
	for( index=0; index<numpnts(morphedAPLatencyWave); index=index+1 )
		counterAP =  floor(morphedAPLatencyWave(index)*1000/APSampInt )
		stimWave[counterAP, 2] = 1			// Make stimulus trigger 2 pts long (100 s @ 50 s/pt).
	endfor

	DoWindow/K MorphedAPPlot					// Close old MorphedAPPlot.
	Display/M/W=(9,19.5,40,23) stimWave	
	DoWindow/C MorphedAPPlot					// Set name (used for closing window).
	SetScale y 0, 0, "norm", stimWave
	
	CalculateDCwaves()							// Calculate wave arguments for dynamic clamp.  Function defined below.

	KillWaves/Z lat, tmpEPSCcycleWave, tmpAPcycleWave		// Clean up.
	DoWindow/F DynamicClamp

End

//_________ Calculate wave arguments for dynamic clamp _________

Function CalculateDCwaves()
	
	Variable startTimePts, endTimePts, index, index_pnts
	Variable DCpeakThresh = 0.5			// 0.5 should catch all interpolated stimuli.
	Variable DCminWidthX = 0.2			// 200 s min width for finding stimuli.
	Variable stimPulseDur = 5			// Duration (in pts) of stim pulse in DIGwave (100 s @ 20 s/pt).

	NVAR DCEpiLen, DCSampInt, BaselineDur, EPSGIsOn, StimIsOn, EPSCwaveFirstOdourResp_index
	NVAR EPSCSampInt, DCEpiLenSecs, E_MeanAmp
	WAVE ExcitWave, InhibWave, DIGWave, stimWave, EPSCwave

	ExcitWave = 0
	InhibWave = 0
	DIGWave = 0
	
	if( StimIsOn == 1 )					// Make stimulator wave.
		Duplicate/O stimWave, tmpstimWave
		startTimePts = EPSCwaveFirstOdourResp_index-round(BaselineDur/(EPSCSampInt*1e-6))
		DeletePoints 0, startTimePts, tmpstimWave
		endTimePts = round(DCEpiLenSecs/(EPSCSampInt*1e-6))
		DeletePoints endTimePts, numpnts(tmpstimWave)-endTimePts, tmpstimWave
		DIGWave = tmpstimWave[p*(numpnts(tmpstimWave)-1)/(numpnts(DIGWave)-1)]
				// Now need to correct for attenuation of stimulus amplitudes due to interpolation.
				// Use inelegant but safe method: again find stim peaks, then redraw them.
		SetScale/P x 0, DCSampInt/1000,"ms", DIGWave
		FindEpiPeaks(DIGWave, DCSampInt, DCPeakThresh, DCMinWidthX)		// Function defined below.
		WAVE lat							// 'lat' is a wave containing peak latencies (in ms) found by 'FindEpiPeaks'.
		DIGWave = 0						// Zero 'DIGWave' then refill it with stimuli.
		for( index=0; index<numpnts(lat); index=index+1 )
			index_pnts =  floor( lat(index)/(DCSampInt*1e-3) )
			DIGWave[index_pnts, stimPulseDur] = 1		// Default stim pulse duration 5 pts long.
		endfor

	endif
	
	if(EPSGIsOn == 1 )						// Make EPSG wave.
		Duplicate/O EPSCwave, tmpEPSCwave
		startTimePts = EPSCwaveFirstOdourResp_index-round(BaselineDur/(EPSCSampInt*1e-6))
		DeletePoints 0, startTimePts, tmpEPSCwave
		endTimePts = round(DCEpiLenSecs/(EPSCSampInt*1e-6))
		DeletePoints endTimePts, numpnts(tmpEPSCwave)-endTimePts, tmpEPSCwave
		ExcitWave = tmpEPSCwave[p*(numpnts(tmpEPSCwave)-1)/(numpnts(ExcitWave)-1)]
		
		WaveStats/Q Excitwave
		Excitwave = Excitwave/V_min		// Normalise amplitude.
		ExcitWave = Excitwave*1e-9		// Convert to Siemens.
		ExcitWave *= E_MeanAmp			// Scale peak to desired nS.
		ExcitWave[DCEpiLen-2, ] = 0		// Zero last 2 points in array for safety.
	endif

	KillWaves/Z lat, tmpstimWave, tmpEPSCwave	// Clean up.

End

//_________ Filters and normalises respiration _________

Function FilterAndNormaliseResp( targetWave )
	WAVE targetWave
		
	Duplicate/O targetWave, tmprespwave
	Smooth/B=200/F 2000, tmprespwave				// Smooth heavily to get low freq component.
	targetWave = targetWave - tmprespwave			// Subtract low freq component - like high-pass filter.
	Smooth/B/F 1000, targetWave						// Smooth less heavily for low-pass filter.
	WaveStats/Q targetWave
	targetWave = (targetWave-V_min)/(V_max-V_min)	// Normalise.
	
End

//_________ Finds peaks in episode (both respiration and APs) _________

Function FindEpiPeaks( targetWave, localSampInt, localRespThresh, localMinWidthX )
	WAVE targetWave
	Variable localSampInt
	Variable localRespThresh
	Variable localminWidthX

	Variable epiCount, index
	Variable startPeakSearch, endPeakSearch
	
	NVAR MaxPeaks						// Maximum number of peaks that can be accommodated.
		
	Make/O/N=(MaxPeaks) lat			// Wave to hold peak latencies.
	lat = 0
		
	FindLevels/M=(localMinWidthX)/Q/EDGE=1/D=lat targetWave, localRespThresh	// Find crossings, save in wave 'lat'.
	
	for( index=0; index<V_LevelsFound; index=index+1 )			// Go back and find peak time for each transient.
		startPeakSearch = lat[index]
		endPeakSearch = startPeakSearch + localMinWidthX
		FindPeak/M=(localRespThresh)/Q/R=(startPeakSearch, endPeakSearch) targetWave
		lat[index] = V_PeakLoc											// Replace level crossing time with peak time.
	endfor
	
	DeletePoints index, MaxPeaks-index, lat							// Delete trailing zeroes.
		
End

//_________ Read AxoGraph file _______________

// This uses data format provided by John Clements, AxoGraph Scientific.

Function ReadAxoGraph()
	
	Variable fileHandle
	Variable formatVersion
	Variable index, numColumns, columnType, columnTitleLength
	Variable scale, zeroPoint, yColCounter
	String tmpStr

	NVAR	EpiLen, SampInt, ChipFlag
	SVAR	ShortFileName
	WAVE/Z dataWave, tmpWave

	Make/O/B/U/N=4 headerIdentifier			// Unsigned 8-bit words (4 bytes long).
		
	Open/R/T="????" fileHandle
	if( fileHandle != 0 )
		FBinRead/B=(ChipFlag)/F=1/U fileHandle, headerIdentifier			// Read unsigned 8-bit word (4 bytes).
		// If this is an OS X file, header should contain {97,120,103,120} which is Ascii {a, x, g, x}.
				
		if( headerIdentifier[0] == 97 )		// OS X data file.
			FSetPos fileHandle, 8				// Read number of columns to follow.
			FBinRead/B=(ChipFlag)/F=3 fileHandle, numColumns		// Read long integer (4 bytes).
													// Find length of first column and assume it's the same for all.
			Make/O/N=3 columnHeader			// Scratch wave into which will repeatedly read column header info.
			FBinRead/B=(ChipFlag)/F=3 fileHandle, columnHeader	// Read long integer.
			EpiLen = columnHeader[0]
													// Predefine 1D wave to read into (FSetPos can't handle 2D waves).
			Make/O/N=(EpiLen) tmpWave
													// Predefine 2D wave to hold all columns of data.
			Make/O/N=(EpiLen,numColumns) dataWave
			FSetPos fileHandle, 12			// Reset pointer back before last FBinRead so code below is same for each column.
			
			for( index=0; index<numColumns; index+=1 )
				FBinRead/B=(ChipFlag)/F=3 fileHandle, columnHeader	// Read long integer.
				EpiLen = columnHeader[0]
				columnType = columnHeader[1]
				columnTitleLength = columnHeader[2]/2						// Following John C's example, simply divide by 2.

				FStatus fileHandle												// Find byte offset for current pointer position.
				FSetPos fileHandle, V_filePos+columnTitleLength*2		// Move pointer past Column Title text (2 bytes/char) to start of data.

				if( columnType == 4 )					// 'short' type of data (i.e. signed 16-bit word, 2 bytes each).
					FBinRead/B=(ChipFlag)/F=2 fileHandle, tmpWave
					dataWave[][index] = tmpWave[p]	// This puts tmpWave into column number 'index' of dataWave.
				endif

				if( columnType == 5 )					// 'long' type of data (i.e. signed 32-bit word, 4 bytes each).
					FBinRead/B=(ChipFlag)/F=3 fileHandle, tmpWave
					dataWave[][index] = tmpWave[p]
				endif

				if( columnType == 6 )					// 'float' type of data (i.e. 32-bit IEEE floating point).
					FBinRead/B=(ChipFlag)/F=4 fileHandle, tmpWave
					dataWave[][index] = tmpWave[p]
				endif

				if( columnType == 7 )					// 'double' type of data (i.e. 64-bit IEEE floating point).
					FBinRead/B=(ChipFlag)/F=5 fileHandle, tmpWave
					dataWave[][index] = tmpWave[p]
				endif

				if( columnType == 9 )					// 'series' type of data.
					Make/O/D/N=2 workArr						// Wave to hold first xData point and sample increment (double).
					FBinRead/B=(ChipFlag)/F=5 fileHandle, workArr		// Read first two xData points as 32-bit IEEE floating point.
					SampInt = workArr[1]*1e6									// Calculate sample interval (convert sec to s).
				endif
		
				if( columnType == 10 )				// 'scaled short' type of data.
					Make/O/D/N=2 workArr						// Wave to hold scale factor and zero point for yData (double).
					FBinRead/B=(ChipFlag)/F=5 fileHandle, workArr		// Read first two points as 32-bit IEEE floating point.
					scale = workArr[0]											// Scale factor.
					zeroPoint = workArr[1]									// Zero point.
					FBinRead/B=(ChipFlag)/F=2 fileHandle, tmpWave		// Read rest of column as signed 16-bit integer.
					tmpWave = 1000*(tmpWave*scale + zeroPoint)			// Scale.
					dataWave[][index] = tmpWave[p]
				endif
			endfor	
		endif
		
		ShortFileName = ParseFilePath(3, S_fileName, ":", 0, 0)		// Extracts file name without path or extension.
		Close fileHandle
	endif

End

//_________ Set baseline duration Variable Control ___________

Function SetBaselineDuration(ctrlName, varNum, varStr, varName) : SetVariableControl
	String ctrlName
	Variable varNum				// This is the set value.
	String varStr
	String varName
	
	NVAR	BaselineDur
	
	BaselineDur = varNum

	CalculateDCwaves()			// Calculate wave arguments for dynamic clamp.
	
End

//_________ Set DC episode duration Variable Control ___________

Function SetDCEpisodeDuration(ctrlName, varNum, varStr, varName) : SetVariableControl
	String ctrlName
	Variable varNum				// This is the set value.
	String varStr
	String varName
	
	NVAR	DCEpiLen, DCSampInt, DCEpiLenSecs
	
	DCEpiLenSecs = varNum
	DCEpiLen = DCEpiLenSecs/(DCSampInt*1e-6)		// This should be divisible by 4.
															// (Get bug if try to check this programmatically.)
	InitDCWaves()
	CalculateDCwaves()			// Calculate wave arguments for dynamic clamp.
	
End

//_________ Select "EPSG ON" Checkbox _______________

Function EPSGOnCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	
	NVAR	EPSGIsOn

	if (checked)
		EPSGIsOn = 1
	else
		EPSGIsOn = 0
	endif
	
	CalculateDCwaves()			// Calculate wave arguments for dynamic clamp.
	
End

//_________ Select "Stimulator ON" Checkbox _______________

Function StimOnCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	
	NVAR	StimIsOn

	if (checked)
		StimIsOn = 1
	else
		StimIsOn = 0
	endif
	
	CalculateDCwaves()			// Calculate wave arguments for dynamic clamp.
	
End

//_________ Select "Plot DC Waves ON" Checkbox _______________

Function PlotDCWavesOnCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	
	NVAR	DCWavesPlotIsOn
	WAVE ExcitWave, DIGWave

	if (checked)
		DCWavesPlotIsOn = 1
		CalculateDCwaves()						// Calculate wave arguments for dynamic clamp.
		DoWindow/K DCWavePlot					// Close old plot.
		Display/M/W=(9,24,40,32.5)			// Make new plot.
		DoWindow/C DCWavePlot					// Set name (used for closing window).
		Display/W=(0,0,1,0.6)/HOST=# ExcitWave
		SetScale y 0, 0, "mV", APwave
		Display/W=(0,0.6,1,1)/HOST=## DIGWave
	else
		DCWavesPlotIsOn = 0
		DoWindow/K DCWavePlot
	endif
	DoWindow/F DynamicClamp
	
End

//_________ Press "Clear All Plots" button  ____________

Function ClearAllPlotsButton(ctrlName) : ButtonControl
	String ctrlName
	
	DoWindow/K EPSCPlot
	DoWindow/K APPlot
	DoWindow/K MorphedAPPlot
	DoWindow/K DCWavePlot
	DoWindow/F DynamicClamp

End

//_________ "Test Pulse" ROUTINES BELOW _____________

// ASSUMPTIONS:

//  Are using an ITC-18 interface (with clock ticks at 1.25 s intervals).

//  The following global variables are defined in calling routine: 
//		VCHeadStageGain, ICHeadStageGain, VCScaledOutputGain, ICScaledOutputGain, ADCRange,
//		TPSampleClockTicks, VCExtCmdSens, ICExtCmdSens, TestPulseFirstVisit, mVtoDig,
//		ITCisConnected, TestPulseVCorIC

//  The following two lines are in the initialisation block of calling routine:
//		DoWindow/K TestPulseCurrent						// Close old Im window.
//		DoWindow/K TestPulseControls					// Close old Test Pulse control window.

//  The following flag is set in the calling routine:
//		TestPulseFirstVisit = 1

//  Resistance is calculated from baseline (0 to one point before step) and (average of last 100 points in step).

//_________ Hit "Test Pulse" Button on main panel _____________

Function TestPulseButton(ctrlName) : ButtonControl
	String ctrlName
	
	NVAR TestPulseFirstVisit
	
	Variable/G TestEpiLen = 1000			// Length of TestPulse wave (points).
	Variable/G TestResistance = 0			// Calculated resistance.
	Variable/G StartTestStep_pts = 200		// Start of test pulse step (points).
	Variable/G DurTestStep_pts = 600		// Duration of test pulse step (points).
	Variable/G TestSampInt					// SampInt (s).
	Variable/G VCTestAmp						// VC test pulse amplitude (mV).
	Variable/G ICTestAmp						// IC test pulse amplitude (pA).
	Variable/G VCCurrentDisplayRange		// Range of display (nA).
	Variable/G ICCurrentDisplayRange		// Not used in current version - autoscale axes in IC.

	if( TestPulseFirstVisit == 1 )
		TestSampInt = 10						// Default SampInt (s).
		VCTestAmp = 10							// Default VC test pulse amplitude (mV).
		ICTestAmp = -100						// Default IC test pulse amplitude (pA).
		VCCurrentDisplayRange = 2			// Default scale means initially 2 nA range.
		
		TestPulseFirstVisit = 0
	endif

	SetupTestWindows()
	StartTestPulse()
	
End

//_________ Set up Test Windows _____________

Function SetupTestWindows()

	NVAR	VCHeadStageGain, VCScaledOutputGain				// Globals in following 4 lines are defined
	NVAR	ICHeadStageGain, ICScaledOutputGain				// by calling routine.
	NVAR	ADCRange, TPSampleClockTicks
	NVAR	TestPulseVCorIC											// (Flag specifying mode: 1 = VC, 2 = IC.)
	NVAR 	TestEpiLen, TestSampInt, VCTestAmp, ICTestAmp	// Globals in following 2 lines are defined
	NVAR 	TestResistance, VCCurrentDisplayRange				// within Test Pulse routine.
	
	// Ensure following two lines are in initialisation block of calling routine:
	
	DoWindow/K TestPulseCurrent								// Close old Im window.
	DoWindow/K TestPulseControls							// Close old Test Pulse control window.
	
	// Define and initialize waves
			// TestResponseWave = Response (Im or Vm); TestCmdWave = Command steps; TestResponseWaveAve = Ave response.
	Make/O/N=(TestEpiLen) TestResponseWave, TestResponseWaveAve, TestCmdWave
	SetScale/P x 0, TestSampInt/1e6,"s", TestResponseWave, TestResponseWaveAve, TestCmdWave		// Set time axes to seconds.
	
	TestResponseWave = 0
	TestResponseWaveAve = 0
	TestCmdWave = 0
	
	Display/W=(250,110,700,465) TestResponseWaveAve	// Define new Im window.
	DoWindow/C TestPulseResponse							// Set name (used for closing window).
	if( TestPulseVCorIC == 1 )
		SetAxis left -VCCurrentDisplayRange*1e-9, VCCurrentDisplayRange*1e-9
		SetScale y 0, 0, "A", TestResponseWaveAve
	elseif( TestPulseVCorIC == 2 )
		SetAxis/A left
		SetScale y 0,0,"V", TestResponseWaveAve
	endif
	
	NewPanel /W=(250,490,700,575) as "TestPulseControls"		// Define new Test Pulse control window.
	DoWindow/C TestPulseControls							// Set name for closing.
	SetDrawLayer UserBack
	SetDrawEnv fsize= 10
	DrawText 20,65,"Press 'Esc' to stop"
	if( TestPulseVCorIC == 1 )								// Voltage Clamp mode.
		ValDisplay testvaldisp0_0,pos={20,19},size={150,22},title="R (M)",fSize=16
		ValDisplay testvaldisp0_0,format="%6.1f",fStyle=1
		ValDisplay testvaldisp0_0,limits={-inf,0,inf},barmisc={0,1000}
		ValDisplay testvaldisp0_0,value= #"TestResistance"
		SetVariable testsetvar0_1,pos={270,10},size={90,15},title="Vp (mV)"
		SetVariable testsetvar0_1,limits={-100,100,10},value= VCTestAmp
		Button testbutton0_2,pos={276,31},size={81,20},proc=ScaleUpDisplay,title="Zoom In"
		Button testbutton0_3,pos={275,56},size={83,21},proc=ScaleDownDisplay,title="Zoom Out"
		
	elseif( TestPulseVCorIC == 2 )							// Current Clamp mode.
		SetVariable testsetvar0_4,pos={270,10},size={90,15},title="Istep (pA)"
		SetVariable testsetvar0_4,limits={-1000,1000,100},value= ICTestAmp
	endif
	
	Button testbutton0_5,pos={380,41},size={45,20},proc=ExitTestPulse,title="Exit"
	Button testbutton0_6,pos={200,41},size={50,20},proc=StartTestPulseButton,title="Start"

	DoWindow/F TestPulseControls

End

//_________ Start Test Pulse Button _____________

Function StartTestPulseButton(ctrlName) : ButtonControl
	String ctrlName

	StartTestPulse()
	
End

//_________ Start Test Pulse _____________

Function StartTestPulse()
	
	NVAR	VCExtCmdSens, StartTestStep_pts, DurTestStep_pts, VCTestAmp, TestResistance
	NVAR	TPSampleClockTicks, TestEpiLen, VCHeadStageGain, VCScaledOutputGain, ICScaledOutputGain
	NVAR	ITCisConnected, TestPulseVCorIC
	WAVE	TestResponseWave, TestResponseWaveAve, TestCmdWave
	
	Variable numberToAverage = 5					// Number of test pulses to average per 'Resistance' calculation.
	Variable keys, baselineAverage, teststepAverage, testcount
	Variable testTempPoint1, testTempPoint2
	
	String ITCCmd
	if( ITCisConnected == 1 )
		Execute "ITC18Reset"							// Switch out of dynamic clamp mode.
		sprintf ITCCmd, "ITC18Seq \"0\" \"0\""	// DAC=0, ADC=0.
		Execute ITCCmd									// (Note method for executing text command within a function).
	endif

	CalculateTestPulseStep (TestPulseVCorIC)
	
	do														// Start of main infinite loop.
		keys = GetKeyState(0)
		if ((keys & 32) != 0)							// User is pressing escape?
			break
		endif
		for( testcount=0; testcount<numberToAverage; testcount = testcount+1 )			// Start of averaging loop.
			if( ITCisConnected == 1 )
				Execute "ITC18Stim TestCmdWave"							// Load stim and tell ITC length of wave to sample.
				Execute "ITC18StartAcq TPSampleClockTicks, 2, 0"	// Start immediately with DAC output.
				Execute "ITC18Samp TestResponseWave"					// This is done after sampling is finished.
				Execute "ITC18StopAcq"									// Must explicitly stop to reinitialise.
			endif
			
			if( TestPulseVCorIC == 1 )		// VC mode.
				TestResponseWave = TestResponseWave*(10.240/32768)/(VCScaledOutputGain*VCHeadStageGain)		// In nA.
				TestResponseWave = TestResponseWave*1e-9																	// In A.
			else									// IC mode.
				TestResponseWave = TestResponseWave*(10.240/32768)/(ICScaledOutputGain)							// In V.
			endif

//			TestResponseWave[0, TestEpiLen-1] = TestCmdWave*5e-11 + gnoise(10e-9)		// For testing.
			
			TestResponseWaveAve = (TestResponseWaveAve*testcount + TestResponseWave)/(testcount+1)

		endfor
		if( TestPulseVCorIC == 1 )			// VC mode.
					// Average baseline from point 0 to one point before start of test pulse.
			testTempPoint1 = 0
			testTempPoint2 = pnt2x( TestResponseWaveAve, StartTestStep_pts - 1)
			baselineAverage = mean( TestResponseWaveAve, testTempPoint1, testTempPoint2 )
					// Average step response over 100 points ending one point before end of test pulse.
			testTempPoint1 = pnt2x( TestResponseWaveAve, StartTestStep_pts + DurTestStep_pts - 100 )
			testTempPoint2 = pnt2x( TestResponseWaveAve, StartTestStep_pts + DurTestStep_pts - 1 )
			teststepAverage = mean( TestResponseWaveAve, testTempPoint1, testTempPoint2 )

			if( teststepAverage - baselineAverage != 0 )
				TestResistance = VCTestAmp*1e-3/(teststepAverage - baselineAverage)		// Volts/Amps.
				TestResistance = TestResistance/1e6					// Convert to MOhm.
			endif
		endif
		DoUpdate
		
	while(1)
	
End

//_________ Calculate Test Pulse step ___________________________

Function CalculateTestPulseStep (localVCorIC)
	Variable localVCorIC
	
	Variable valToDig, localTestAmp
	
	NVAR	VCExtCmdSens, StartTestStep_pts, DurTestStep_pts, TestSampInt, VCTestAmp, ICExtCmdSens, ICTestAmp
	WAVE	TestCmdWave
		
	TestCmdWave = 0
	
	if( localVCorIC == 1 )
		valToDig = (32768/10240)*(1000/VCExtCmdSens)		// Voltage scale factor.
		localTestAmp = VCTestAmp
	elseif( localVCorIC == 2 )
		valToDig = (32768/10240)*(1000/ICExtCmdSens)		// Current scale factor.
		localTestAmp = ICTestAmp
	endif
	
	SetScale/P x 0,TestSampInt/1000, TestCmdWave			// Set x-axis scale to milliseconds.
	TestCmdWave[ StartTestStep_pts, StartTestStep_pts+DurTestStep_pts ] = localTestAmp*valToDig
	
End

//_________ Scale Up Display Button (i.e. reduce display range) ___________________________

Function ScaleUpDisplay(ctrlName) : ButtonControl
	String ctrlName
	
	NVAR	VCCurrentDisplayRange
	
	VCCurrentDisplayRange = VCCurrentDisplayRange/2.
	
	DoWindow/F TestPulseCurrent				// Switch to test pulse display window.
	SetAxis left -VCCurrentDisplayRange*1e-9, VCCurrentDisplayRange*1e-9
	DoWindow/F TestPulseControls			// Switch to test pulse control window.
End

//_________ Scale Down Display Button (i.e. increase display range) ___________________________

Function ScaleDownDisplay(ctrlName) : ButtonControl
	String ctrlName
	
	NVAR	VCCurrentDisplayRange
	
	VCCurrentDisplayRange = VCCurrentDisplayRange*2
	
	DoWindow/F TestPulseCurrent				// Switch to test pulse display window.
	SetAxis left -VCCurrentDisplayRange*1e-9, VCCurrentDisplayRange*1e-9
	DoWindow/F TestPulseControls			// Switch to test pulse control window.
End

//_________ Exit Test Pulse Button _____________

Function ExitTestPulse(ctrlName) : ButtonControl
	String ctrlName
	
	WAVE	TestResponseWave, TestResponseWaveAve, TestCmdWave

	DoWindow/K TestPulseResponse			// Close old response window.
	DoWindow/K TestPulseControls			// Close old Test Pulse control window.
	
	DoWindow/F ControlPanel
	
	KillWaves/Z TestResponseWave, TestResponseWaveAve, TestCmdWave
	
End
